﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapCareSiteToCrm : MapperRelatedBase
    {
        public MapCareSiteToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper) 
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_caresite> MapInsert(CareSite careSite, Account provider)
        {
            return await MapInsert(careSite);
        }

        public async Task<ppms_caresite> MapInsert(CareSite careSite)
        {
            ppms_caresite entity = null;

            // map entity
            entity = new ppms_caresite
            {
                Id = Guid.NewGuid(),
                ppms_name = FormatName(careSite.Name),
                ppms_othername = careSite.OtherName,
                ppms_isexternal = true,
                ppms_ishandicapaccessible = careSite.IsHandicapAccessibleSpecified && careSite.IsHandicapAccessible,
                ppms_sitetype = EnumHelper.MapEnumToOptionSetValue<ppms_caresite_ppms_sitetype>(careSite.SiteType.ToString())
            };

            // VISN
            if (!string.IsNullOrWhiteSpace(careSite.Visn))
            {
                var ppmsVisn = await GetVisn(careSite.Visn);
                if (ppmsVisn != null)
                {
                    entity.ppms_visnid = ppmsVisn.ToEntityReference();
                }
            }

            // address
            if (careSite.SiteAddress != null)
            {
                // set name value
                if (string.IsNullOrEmpty(careSite.Name)) entity.ppms_name = careSite.SiteAddress.CompositeId;
                entity.ppms_address_line1 = $"{careSite.SiteAddress.Address1} {careSite.SiteAddress.Address2} {careSite.SiteAddress.Address3}".Trim();
                entity.ppms_address_city = careSite.SiteAddress.City;
                entity.ppms_address_postalcode = careSite.SiteAddress.PostalCode;
                entity.ppms_address_county = careSite.SiteAddress.County;
                entity.ppms_address_country = careSite.SiteAddress.CountryCode;

                // lookup state reference
                if (!string.IsNullOrEmpty(careSite.SiteAddress.State))
                {
                    var vaState = GetState(careSite.SiteAddress.State);
                    if (vaState != null) entity.ppms_address_state = vaState;
                }
            }

            // Center of excellence
            if (careSite.CenterOfExcellenceSpecified)
            {
                entity.ppms_CenterofExcellence = EnumHelper.MapEnumToOptionSetValue<ppms_caresite_ppms_CenterofExcellence>(careSite.CenterOfExcellence.ToString());
            }

            // Place of Service Codes
            if (careSite.PlaceOfServiceCodes != null && careSite.PlaceOfServiceCodes.Item != null)
            {
                var pos = new List<ppms_providerplaceofservice>();
                foreach (var item in careSite.PlaceOfServiceCodes.Item)
                {
                    pos.Add(new ppms_providerplaceofservice()
                    {
                        ppms_PlaceofServiceCodeId = GetPlaceOfServiceCode(item.Code),
                        ppms_CareSiteId = entity.ToEntityReference()
                    });
                }

                if (pos.Count > 0) entity.ppms_ppms_caresite_ppms_providerplaceofservice = pos;
            }

            //Phone
            if (!string.IsNullOrWhiteSpace(careSite.Phone)) entity.ppms_mainsitephone = careSite.Phone;
            //Fax
            if (!string.IsNullOrWhiteSpace(careSite.Fax)) entity.ppms_CareSiteFax = careSite.Fax;
            //Email
            if (!string.IsNullOrWhiteSpace(careSite.Email)) entity.ppms_CareSiteEmail = careSite.Email;

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            return entity;
        }

        public async Task<ppms_caresite> MapUpdate(CareSite entity)
        {
            // get entity
            var crmEntity = await CheckForExistingEntity(entity);
            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity);
            }

            return await MapUpdate(entity, crmEntity);
        }

        public async Task<ppms_caresite> MapUpdate(CareSite entity, Account parent)
        {
            var crmEntity = GetCrmEntity(parent, entity);
            if (crmEntity == null)
                return await MapUpdate(entity);
            else
                return await MapUpdate(entity, crmEntity);
        }

        public async Task<ppms_caresite> MapUpdate(CareSite entity, ppms_caresite crmEntity)
        {
            if (crmEntity == null)
            {
                HandleException("Invalid parameter provided (ppms_caresite)");
                return null;
            }

            var newEntity = new ppms_caresite()
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name,
                ppms_othername = crmEntity.ppms_othername,
                ppms_ishandicapaccessible = crmEntity.ppms_ishandicapaccessible,
                ppms_sitetype = crmEntity.ppms_sitetype
            };

            // Map fields
            if (IsChanged(entity.OtherName, crmEntity.ppms_othername)) newEntity.ppms_othername = entity.OtherName;

            if (entity.IsHandicapAccessibleSpecified && entity.IsHandicapAccessible != crmEntity.ppms_ishandicapaccessible) newEntity.ppms_ishandicapaccessible = entity.IsHandicapAccessible;

            // VISN
            if (!string.IsNullOrEmpty(entity.Visn))
            {
                var ppmsVisn = await GetVisn(entity.Visn);
                if (ppmsVisn != null)
                {
                    crmEntity.ppms_visnid = ppmsVisn.ToEntityReference();
                }
            }

            // Address
            if (entity.SiteAddress != null)
            {
                // set name value
                if (!string.IsNullOrEmpty(entity.Name) && IsChanged(newEntity.ppms_name, entity.Name))
                    newEntity.ppms_name = entity.Name;
                newEntity.ppms_address_line1 = $"{entity.SiteAddress.Address1} {entity.SiteAddress.Address2} {entity.SiteAddress.Address3}".Trim();
                newEntity.ppms_address_city = entity.SiteAddress.City;
                newEntity.ppms_address_postalcode = entity.SiteAddress.PostalCode;
                newEntity.ppms_address_county = entity.SiteAddress.County;
                newEntity.ppms_address_country = entity.SiteAddress.CountryCode;

                // lookup state reference
                if (!string.IsNullOrEmpty(entity.SiteAddress.State))
                {
                    var vaState = GetState(entity.SiteAddress.State);
                    if (vaState != null) newEntity.ppms_address_state = vaState;
                }
            }

            if (entity.SiteType.ToString() != crmEntity.ppms_sitetype.ToString())
            {
                newEntity.ppms_sitetype = EnumHelper.MapEnumToOptionSetValue<ppms_caresite_ppms_sitetype>(entity.SiteType.ToString());
            }

            // Center of excellence
            if (entity.CenterOfExcellenceSpecified)
            {
                newEntity.ppms_CenterofExcellence = EnumHelper.MapEnumToOptionSetValue<ppms_caresite_ppms_CenterofExcellence>(entity.CenterOfExcellence.ToString());
            }

            // Place of Service Codes
            if (entity.PlaceOfServiceCodes != null && entity.PlaceOfServiceCodes.Item != null)
            {
                var pos = new List<ppms_providerplaceofservice>();
                EntityReference targetCode;

                foreach (var item in entity.PlaceOfServiceCodes.Item)
                {
                    targetCode = GetPlaceOfServiceCode(item.Code);

                    if (targetCode != null &&
                        crmEntity.ppms_ppms_caresite_ppms_providerplaceofservice != null &&
                        !crmEntity.ppms_ppms_caresite_ppms_providerplaceofservice.Any(x => x.ppms_providerplaceofserviceId == targetCode.Id))
                    {
                        pos.Add(new ppms_providerplaceofservice()
                        {
                            ppms_PlaceofServiceCodeId = targetCode,
                            ppms_CareSiteId = newEntity.ToEntityReference()
                        });
                    }
                }

                if (pos.Count > 0) newEntity.ppms_ppms_caresite_ppms_providerplaceofservice = pos;
            }

            //Phone
            if (IsChanged(entity.Phone, crmEntity.ppms_mainsitephone)) newEntity.ppms_mainsitephone = entity.Phone;
            //Fax
            if (IsChanged(entity.Fax, crmEntity.ppms_CareSiteFax)) newEntity.ppms_CareSiteFax = entity.Fax;
            //Email
            if (IsChanged(entity.Email, crmEntity.ppms_CareSiteEmail)) newEntity.ppms_CareSiteEmail = entity.Email;

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            return newEntity;
        }

        private ppms_caresite GetCrmEntity(Account provider, CareSite careSite)
        {
            Address address = null;
            IEnumerable<ppms_caresite> list = null;

            if (careSite != null) address = careSite.SiteAddress;

            if (base.ParentalReference != null)
            {
                var source = (Account)ParentalReference;
                if (source != null)
                    list = source.ppms_account_caresite_organization;
            }

            if (list == null)
                list = provider.ppms_account_caresite_organization;

            // Check required parameters
            if (list == null || address == null) return null;

            var entityList = list as ppms_caresite[] ?? list.ToArray();

            string name = FormatName(careSite.Name);
            return entityList?
                .Where(a => a.ppms_name == name)?
                .FirstOrDefault(x => x.ppms_address_compositeid == address.CompositeId);
        }

        private async Task<ppms_caresite> CheckForExistingEntity(CareSite careSite)
        {
            var address = careSite.SiteAddress;
            if (address == null) return null;

            string name = FormatName(careSite.Name);
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var entityList = context.ppms_caresiteSet.Where(a => a.ppms_name == name);
                var entity = entityList.FirstOrDefault(e => e.ppms_address_compositeid == address.CompositeId);
                context.LoadProperty(entity, new Relationship("ppms_ppms_caresite_ppms_providerplaceofservice"));
                return entity;
            }
        }

        private string FormatName(string name)
        {
            string returnValue = name;

            if (!string.IsNullOrWhiteSpace(name))
            {
                if (name.Length > 100)
                    returnValue = name.Substring(0, 100);
            }
            return returnValue;
        }

        private static CareSite ConvertEntity<T>(T entity)
        {
            return (CareSite)Convert.ChangeType(entity, typeof(CareSite));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (entities == null) return;

            if (IsWithinContext)
            {
                bool isAddingRelationship;

                // Make sure provider entity is being tracked
                if (!Context.IsAttached(parent)) Context.Attach(parent);

                IEnumerable<ppms_caresite> children = null;
                var account = (Account)ParentalReference;
                if (account != null)
                {
                    children = account.ppms_account_caresite_organization;
                }

                foreach (var item in entities)
                {
                    isAddingRelationship = true;

                    // If parental reference exists, make sure the child is not associated to parent
                    if (children != null && children.Any(a => a.Id == item.Id))
                    {
                        isAddingRelationship = false;
                        if (!Context.IsAttached(item))
                        {
                            Context.Attach(item);
                            Context.UpdateObject(item);
                        }
                    }

                    if (isAddingRelationship)
                    {
                        Context.AddRelatedObject(parent, new Relationship("ppms_account_caresite_organization"), item);

                        // Add related place of service
                        AddRelatedEntities(item, "ppms_ppms_caresite_ppms_providerplaceofservice");
                    }
                }
            }
            else
            {
                if (entities.Count <= 0) return;

                var account = (Account)parent;
                if (account != null) account.ppms_account_ppms_providerservice = ConvertEntityList<ppms_providerservice>(entities);
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_caresite>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var matches = provider.ppms_account_caresite_organization.Where(p => p.Id == new Guid(entity.CorrelationId));
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_caresite>)list);
        }
    }
}